#!/usr/bin/env python3
# -*- coding:utf-8 -*-
###
# File: /data/zhangruochi/projects/molgen/dataloaders/egrnn_dataloader.py
# Project: /mnt/data/zhangruochi/DeepOrchestration/dataloader
# Created Date: Tuesday, October 5th 2021, 4:38:41 pm
# Author: Ruochi Zhang
# Email: zrc720@gmail.com
# -----
# Last Modified: Mon Sep 29 2025
# Modified By: Ruochi Zhang
# -----
# Copyright (c) 2021 Bodkin World Domination Enterprises
#
# MIT License
#
# Copyright (c) 2021 Ruochi Zhang Ltd
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the "Software"), to deal in
# the Software without restriction, including without limitation the rights to
# use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
# of the Software, and to permit persons to whom the Software is furnished to do
# so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# -----
###
import os
import sys

sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from omegaconf import DictConfig
from typing import Union
from pathlib import Path
from typing import Dict
import numpy as np
from torch.utils.data import DataLoader
from ..dataset.mimic_dataset import TabularDataset
from ..utils.preprocessing import preprocess_t_mimic


def dataloaders_preperation(cfg: DictConfig, orig_cwd: Union[Path,
                                                             str]) -> Dict:

    dataset_path = Path(
        os.path.join(orig_cwd, cfg.data.data_path, cfg.data.task_name))
    cache_dir = Path(os.path.join(orig_cwd, cfg.data.cache_path))

    X_train, y_train, train_names, X_valid, y_valid, valid_names, X_test, y_test, test_names = preprocess_t_mimic(
        dataset_path, cache_dir,
        os.path.join(os.path.dirname(__file__), cfg.data.normalizer_state))

    demo_path = os.path.join(orig_cwd, cfg.data.demo_path)

    train_dataset = TabularDataset(X_train, y_train, train_names, demo_path)
    valid_dataset = TabularDataset(X_valid, y_valid, valid_names, demo_path)
    test_dataset = TabularDataset(X_test, y_test, test_names, demo_path)

    dataloaders = {
        "train":
        DataLoader(dataset=train_dataset,
                   batch_size=cfg.train.batch_size,
                   shuffle=True),
        "valid":
        DataLoader(dataset=valid_dataset,
                   batch_size=cfg.train.batch_size,
                   shuffle=False),
        "test":
        DataLoader(dataset=test_dataset,
                   batch_size=cfg.train.batch_size,
                   shuffle=False)
    }

    return dataloaders
